(* An abstraction for a full duplex connected stream

   Only non-blocking sockets are used.

   with this system, the caller (= the module consumer) is responsible for:
     - clearing the read buffer
     - when to close a transaction
   for bandwidth management, this module will accept new data ( ie call read() ) whenever the isreadreadycallback responds true 
   
   
   can be compiled with: bee -o hello -I ~/".opt/bee/standard library" preludedefinitions.b lists.b strings.b printf.b int64.b buffers.b posix.bi tcptransactions.b
   
   TODO: clean up and improve 
   
   *)
   
module Tcptransaction =

type transactionstatus = On | Off
type transactionstate = Connecting | Connected | Terminated | Failed

(* explanation for the readdatacallback type:
   this closure must have the information of "transaction ID" available in some way or another, either passed by argument, or within the closure, 
   (1) by argument: rather simple, but now the type of transaction is recursive (which is not an issue)
   (2) at closure creation: a transaction reference must be given at the closure creation, but since this field is immutable, the closure must be created before the transaction... the circular dependency prevents us to do that, so it's not possible
   As a result, we pass the transaction ID as an argument (option (1)). We hope it's still quite readable. *)
   
type transaction = 
    {
      mutable socket : option int; (* a file descriptor TODO: this does not need to be exposed *)
      currentpartialsent : Buffer.t;
      currentread : Buffer.t; (* the manager is responsible for cleaning up this buffer, but this buffer cannot grow bigger than maxbuffersize *)
      readdatacallback : transaction -> string -> int -> unit; (* the argument is not just any transaction, it's the current transaction, i don't know how to enforce this with the type system *)
      isreadreadycallback : transaction -> bool; (* callback to the manager that will tell whether he can accept incoming data *)
      writequeue : list string;
      (*writequeuelength : int; *) (* TODO: we could store this information too, to make getting the length constant time *)
      mutable transactionstate : transactionstate;
      mutable bytesread : int64; (* only for debugging purposes, remove *)
      timecreated : int64 * int;
    }

let timeoutConnection = 5.0
let maxbuffersize = 1024 * 1024 (* 1Mo *)
let tcpsendsize = 16384
let maxwritequeuesize = 100 (* some limit on our internal write queues size, this means we only buffer 1.6M *)

let buf = String.build 16384

let transactions = ref []

let fail () = 
  let () = printf "fatal error in tcptransactions\n" in 
  Posix.exit 1 (* a fatal error, should never happen *)

let closeandclean trans = (*assert(printf "%f, closing transaction in tcptransaction\n" (Unix.gettimeofday ()); true);*)
  let () = case trans.socket  
   | Some sock -> 
       let ret = Posix.close sock in 
       trans.socket <- None
   | None -> ()
  end in 
  transactions := List.filter (fun tr -> not (tr == trans)) !transactions

let read_handler rsock = (* when writing a consumer, make sure all the callbacks close the transaction when len=0 - how to enforce this? *)
  let () = printf "in tcp read_handler\n" in 
  let n = Posix.recv rsock buf 0 16384 in
  let l = List.filter (fun tr -> case tr.socket | Some s -> s = rsock && tr.transactionstate <> Terminated && tr.transactionstate <> Failed | None -> false end) !transactions in 
  case l
  | [] -> (* this happens if the transaction was closed during a writehandler immediately before *)
      List.iter (fun tr -> case tr.socket | Some s -> if s = rsock then tr.readdatacallback tr "" 0 else () | None -> () end) !transactions
  | trans :: _ -> (* TODO we should assert() whether the rest of the list is empty *)
     if n > 0 then 
       ( (*assert(printf "%f, %i bytes read in tcptransaction\n" (Unix.gettimeofday ()) n; true);*)
        let () = Buffer.add_substring trans.currentread buf 0 n in 
        if Buffer.length trans.currentread > maxbuffersize then 
          ((*assert(printf "buffer too big in tcptransaction\n"; true);*)
           let () = trans.transactionstate <- Failed in
           trans.readdatacallback trans "" 0)
        else
          (let () = trans.bytesread <- Int64.add trans.bytesread (Int64.of_int n) in
           (* TODO: le n dans le callback est redondant TODO: not a good interface *)
           trans.readdatacallback trans (Buffer.contents trans.currentread) (Buffer.length trans.currentread)) )
     else if n = 0 then 
       ((*printf "packet of length 1 received in readhandler in tcptransaction, transaction terminated\n";*)
        let () = trans.transactionstate <- Terminated in
        trans.readdatacallback trans "" 0)
     else (* n < 0 *)
       ((*printf "error in readhandler, failing connection in readhandler in tcptransaction\n";*)
        let () = trans.transactionstate <- Failed in
        trans.readdatacallback trans "" 0(* TODO: better to send back -1? *))
  end

let write_handler wsock = 
  let l = List.filter (fun tr -> case tr.socket | Some s -> s = wsock && tr.transactionstate <> Terminated && tr.transactionstate <> Failed | None -> false end) !transactions in 
  case l  
  | [] -> () (* cannot happen TODO: crash *)
  | trans :: _ -> 
    let () = trans.transactionstate <- Connected in 
    let data = 
      if Buffer.length trans.currentpartialsent > 0 then 
        Buffer.contents trans.currentpartialsent 
      else 
        case trans.writequeue 
        | [] -> "" (* cannot happen, TODO: crash *)
        | buffer :: q -> 
            let () = trans.writequeue <- q in
            buffer
        end in 
    let n = Posix.send wsock data 0 (String.length data) in 
    if n < 0 then 
      trans.transactionstate <- Failed
    else
      let () = Buffer.clear trans.currentpartialsent in
      if n < String.length data then 
        Buffer.add_substring trans.currentpartialsent data n (String.length data - n)
      else
       ()
  end

let senddata data trans = (* TODO: control if the writequeue is full *)
  trans.writequeue <- trans.writequeue @ [data] 
  (*trans.writequeue <- trans.writequeue @ (Utils.cutinsmalltcppieces data tcpsendsize)*)

let iswritequeuefull trans = 
  List.length trans.writequeue > maxwritequeuesize

let gettransactionstatus trans = 
  case trans.transactionstate 
    | Terminated -> Off
    | Failed -> Off
    | Connecting -> On
  end

let getAllSocketsForSelect () = 
  let rsockets = List.map (fun trans -> case trans.socket | Some s -> s | None -> fail () end) 
                   (List.filter (fun trans -> case trans.socket | Some s -> true | None -> false end && 
                                              trans.isreadreadycallback trans) !transactions) in
  let wsockets = List.map (fun trans -> case trans.socket | Some s -> s | None -> fail () end) 
                   (List.filter (fun trans -> (trans.transactionstate = Connecting || (trans.transactionstate <> Failed && trans.transactionstate <> Terminated && 
                                                                                           (Buffer.length trans.currentpartialsent > 0 || trans.writequeue <> []))) && 
                                               case trans.socket | Some s -> true | None -> false end) !transactions) in
  (rsockets, wsockets)
  
let getAllSockets () = 
  let rec filtersockets l = 
    case l
    | [] -> []
    | trans :: q -> case trans.socket 
                    | Some s -> let isread = trans.isreadreadycallback trans in 
                                let iswrite = trans.transactionstate = Connecting || 
                                                 (trans.transactionstate <> Failed && trans.transactionstate <> Terminated && 
                                                  (Buffer.length trans.currentpartialsent > 0 || trans.writequeue <> [])) in 
                                if isread && iswrite then (s, [| Posix.POLLIN; Posix.POLLOUT |]) :: filtersockets q 
                                else if isread then (s, [| Posix.POLLIN |]) :: filtersockets q
                                else if iswrite then (s, [| Posix.POLLOUT |]) :: filtersockets q
                                else filtersockets q                   
                    | None -> filtersockets q 
                    end 
    end in 
  filtersockets !transactions
  

let isSocketHere sock = 
  List.exists (fun trans -> case trans.socket | Some s -> s = sock | None -> false end) !transactions                      
                         
let createTransaction   sockdomain 
                        localaddr 
                        remoteaddr 
                        readdatacallback 
                        isreadreadycallback = 
  let socket = Posix.socket sockdomain Posix.SOCK_STREAM 0 in (* TODO: should catch the case socket < 0 *)
  let ret_bind = Posix.bind socket localaddr in (* TODO: return code must be checked, many error cases *)
  let flags = Posix.fcntl socket Posix.F_GETFL 0 in (* get the flags associated with the socket, we don't check if flags < 0, there is no error case *)
  let _ = Posix.fcntl socket Posix.F_SETFL (flags + Posix.int_of_flag Posix.O_NONBLOCK) in (* set the socket non-blocking, no need to check the return code *)
  let trans = { socket = Some socket;
                currentpartialsent = Buffer.build 8192; 
                currentread = Buffer.build 8192; 
                readdatacallback = readdatacallback; 
                isreadreadycallback = isreadreadycallback; 
                transactionstate = Connecting; 
                writequeue = []; 
                bytesread = 0L; 
                timecreated = Posix.clock_gettime_monotonic (); } in 
  (* assert((match remoteaddr with 
                  Posix.ADDR_INET(x, port) -> printf "making tcp connection to %s:%i in tcptransaction\n" (Posix.string_of_inet_addr x) port 
    | _ -> ()); true); *)
  let ret = Posix.connect socket remoteaddr in 
  
  (*
     Unix.Unix_error ((Unix.EINPROGRESS | Unix.EWOULDBLOCK), "connect", _) -> ()
   | Unix.Unix_error((Unix.ENETUNREACH | Unix.ENETDOWN), "connect", _) -> () (* TODO *)
   | ex -> 
     (assert(printf "%f, Unix.connect failed\n" (Unix.gettimeofday ()); true);
      trans.transactionstate <- Failed));*)
  let () = transactions := trans :: !transactions in
  Some trans
  
  
endmodule
